#!/usr/bin/env python3
import json, os, sys, importlib.util
from pathlib import Path
from datetime import datetime, timezone

HERE = Path(__file__).resolve().parent
ENGINE_PATH = HERE / "engine_i25_kernel_rng.py"
MANIFEST_PATH = HERE / "config" / "manifest_i25.json"

ts = datetime.now(timezone.utc).strftime("%Y-%m-%dT%H-%M-%SZ")
OUTDIR = HERE / "I25_kernel_rng" / ts

def main():
    OUTDIR.mkdir(parents=True, exist_ok=True)

    spec = importlib.util.spec_from_file_location("engine_i25_kernel_rng", ENGINE_PATH)
    engine = importlib.util.module_from_spec(spec)
    spec.loader.exec_module(engine)

    engine.ensure_dirs(str(OUTDIR), ["config","outputs/metrics","outputs/audits","outputs/run_info","logs"])

    with open(MANIFEST_PATH, "r", encoding="utf-8") as f:
        manifest = json.load(f)

    engine.dump_json(str(OUTDIR / "config" / "manifest_i25.json"), manifest)
    engine.write_text(str(OUTDIR / "logs" / "env.txt"),
                      "utc={}\nos={}\npython={}\n".format(ts, os.name, sys.version.split()[0]))

    audit = engine.run_i25(manifest, str(OUTDIR))

    with open(OUTDIR / "outputs" / "run_info" / "result_line.txt", "r", encoding="utf-8") as f:
        print(f.read().strip())

    sys.exit(0 if audit.get("passed") else 1)

if __name__ == "__main__":
    main()